/* global describe it before ethers */

import { expect } from 'chai';
import { ethers } from 'hardhat';
import { Signer } from 'ethers';
// Import contract types
import {
  TradingManagementStorageFacet,
  TradingManagementExecutorFacet,
  DiamondLoupeFacet,
} from '../../../types';

// Add this import for the matcher

// Import Diamond utilities
import { deployDiamond } from '../../scripts/deployTradingManagement';
import {
  deployContractsForTests,
  FameContracts,
  getSigners,
} from './helpers/deploy-helper';
import {
  initializeContracts,
  initializeTradingManagmentContract,
} from './helpers/contracts-initialization-helper';
import { Signers } from './types';
import { upgradeTransparentProxyContract } from '../../scripts/upgradeTransparentProxyContract';
// eslint-disable-next-line @nx/enforce-module-boundaries
import { getIDHashHelper } from '../../../../shared-utils/src/helpers/token-id.utils';

describe('91 - Purchase Asses Rights Subscription Integration Test', function () {
  // Contract instances
  let tradingManagementStorageFacet: TradingManagementStorageFacet;
  let tradingManagementExecutorFacet: TradingManagementExecutorFacet;
  let contracts: FameContracts;
  let signers: Signers;
  let diamondLoupeFacet: DiamondLoupeFacet;

  // Signers
  let owner: Signer;
  let customer: Signer;
  let dataProvider: Signer;

  // Constants
  const assetid = 'aabc';
  const oid = 'oabc';
  const accessRightDuration = 31536000; // duration one year
  const assetPrice = ethers.utils.parseUnits('100', 18); // 100 payment tokens
  const initialPaymentTokenBalance = ethers.utils.parseUnits('1000', 18);
  const tokenId = getIDHashHelper(oid);

  before(async function () {
    // Get signers
    signers = await getSigners();
    owner = signers.admin;
    customer = signers.signer1;
    dataProvider = signers.signer2;

    // Deploy the Diamond contract
    contracts = await deployContractsForTests({ shouldResetNetwork: false });
    await initializeContracts({ contracts, signers });
    const tradingManagementAddress = await deployDiamond({
      saveInfo: false,
      showInfo: false,
    });
    await initializeTradingManagmentContract({
      contracts,
      signers,
      tradingManagementAddress,
    }).then((res) => {
      tradingManagementStorageFacet = res.tradingManagementStorageFacet;
      tradingManagementExecutorFacet = res.tradingManagementExecutorFacet;
      diamondLoupeFacet = res.diamondLoupeFacet;
    });

    // Mint payment tokens to the customer
    await contracts.governanceContract
      .connect(owner)
      .mintCoin(await customer.getAddress(), 'FDE', initialPaymentTokenBalance);

    // Mint an asset to the data provider in the OfferingToken contract

    const result = await contracts.offeringTokenContract
      .connect(owner)
      .addAsset(
        assetid,
        oid,
        'https://example.com/token-uri-3',
        await dataProvider.getAddress(),
        assetPrice,
        accessRightDuration,
        '10',
        '100MB',
        'target1',
        'sl1',
        '0x00',
      );

    await result.wait();
    expect(result).to.be.ok;
  });

  describe('Purchase access right tests', function () {
    before(async function () {
      // Customer approves Bourse to spend their payment tokens
      const approveResult = await contracts.paymentTokenContract
        .connect(customer)
        .approve(contracts.bourseContract.address, initialPaymentTokenBalance);

      await approveResult.wait();
    });

    it('Customer purchases access right (Subscription)', async function () {
      const data = ethers.utils.toUtf8Bytes('Sample data for Subscription');
      const purchaseTime = Math.floor(Date.now() / 1000);
      const offeringTokenIndx =
        await contracts.offeringTokenContract.getIDHash(oid);
      await expect(
        tradingManagementExecutorFacet
          .connect(customer)
          .purchaseAccessRightSubscription(oid, data, accessRightDuration),
      )
        .to.emit(tradingManagementExecutorFacet, 'DataAccessPurchased')
        .withArgs(
          await customer.getAddress(),
          offeringTokenIndx,
          ethers.utils.hexlify(data),
        );

      // // Check that the customer now owns the data access token
      const balance = await contracts.dataAccessSubscriptionContract.balanceOf(
        await customer.getAddress(),
        offeringTokenIndx,
      );
      expect(balance).to.equal(1);

      // // Check payment token balances
      const customerBalanceAfterPurchase =
        await contracts.paymentTokenContract.balanceOf(
          await customer.getAddress(),
        );
      expect(customerBalanceAfterPurchase).to.equal(
        initialPaymentTokenBalance.sub(assetPrice),
      );
      const dataProviderBalanceAfterPurchase =
        await contracts.paymentTokenContract.balanceOf(
          await dataProvider.getAddress(),
        );

      expect(dataProviderBalanceAfterPurchase).to.equal(assetPrice);

      const customerAddress = await customer.getAddress();
      const expirationTime =
        await tradingManagementExecutorFacet.getLastDataAccessSubscriptionExpirationTime(
          customerAddress,
          tokenId,
        );
      expect(expirationTime.toNumber()).to.be.greaterThanOrEqual(
        purchaseTime + accessRightDuration,
      );

      const allExpirationTimes =
        await tradingManagementExecutorFacet.getAllDataAccessExpirationsSubscription(
          customerAddress,
          tokenId,
        );
      expect(allExpirationTimes.length).to.equal(1);
      expect(expirationTime).to.equal(allExpirationTimes[0]);

      it.skip('Owner can burn access token', async function () {
        // Owner calls burnAccessToken
        const data = ethers.utils.toUtf8Bytes('Sample data for burn');
        await expect(
          tradingManagementExecutorFacet
            .connect(owner)
            .burnAccessToken(await customer.getAddress(), oid, data),
        )
          .to.emit(tradingManagementExecutorFacet, 'DataAccessBurned')
          .withArgs(await customer.getAddress(), '', data);

        // Check that the customer's access token is burned
        const offeringTokenIndx =
          await contracts.offeringTokenContract.getIDHash(oid);
        const balance =
          await contracts.dataAccessSubscriptionContract.balanceOf(
            await customer.getAddress(),
            offeringTokenIndx,
          );
        expect(balance).to.equal(0);
      });

      describe('Revert tests', function () {
        it('Should revert if customer has insufficient payment token balance', async function () {
          // Transfer customer's payment tokens back to the owner

          const customerBalance =
            await contracts.paymentTokenContract.balanceOf(
              await customer.getAddress(),
            );

          await contracts.paymentTokenContract
            .connect(customer)
            .transfer(await owner.getAddress(), customerBalance);

          // Attempt to purchase access right
          const data = ethers.utils.toUtf8Bytes('Sample data');
          await expect(
            tradingManagementExecutorFacet
              .connect(customer)
              .purchaseAccessRightPAYG(oid, data, accessRightDuration),
          ).to.be.revertedWith(
            'Insufficient payment token balance to purchase access rights!',
          );
        });

        it('Should revert if asset does not exist', async function () {
          const data = ethers.utils.toUtf8Bytes('Sample data');
          await expect(
            tradingManagementExecutorFacet
              .connect(customer)
              .purchaseAccessRightPAYG(
                'non-existent-oid',
                data,
                accessRightDuration,
              ),
          ).to.be.revertedWith("This offer doesn't exist!");
        });
      });
    });
  });

  describe('Upgrade contract tests', function () {
    it('should upgrade contract without losing token data', async function () {
      // Check initial token
      const offeringTokenIndx =
        await contracts.offeringTokenContract.getIDHash(oid);
      const balance = await contracts.dataAccessSubscriptionContract.balanceOf(
        await customer.getAddress(),
        offeringTokenIndx,
      );
      expect(balance).to.equal(1);
      expect(await contracts.dataAccessSubscriptionContract.ver()).to.equal(1);

      // Upgrade contract
      await upgradeTransparentProxyContract({
        contractName: 'DataAccessV2',
        proxyContractAddress: contracts.dataAccessSubscriptionContract.address,
      });

      // Check that the customer still owns the data access token
      const balance2 = await contracts.dataAccessSubscriptionContract.balanceOf(
        await customer.getAddress(),
        offeringTokenIndx,
      );
      expect(balance2).to.equal(1);
      expect(await contracts.dataAccessSubscriptionContract.ver()).to.equal(2);
    });
  });
});
